#include <stdlib.h>
#include <string.h>
#include "global.h"
#include "field.h"
#include "agent.h"
#include "namespace.h"
#include "utils.h"
#include "routine.h"
#include "evolve.h"

NAMESPACE fields;


int add_field(char * name, FIELD * f)
{
char *s;
if(fields.names_free>=fields.names_size+1)expand_namespace(&fields);
fields.data[fields.names_free]=f;
fields.names[fields.names_free]=strdup(name);
fields.names_free++;
s=strdup(name);
if(f->name!=NULL)free(f->name);
f->name=s;
return 0;
}

void delete_field(long i)
{
long k;
fields.names_free--;
free(fields.names[i]);
free_field(fields.data[i]);
for(k=i;k<fields.names_free;i++){
	fields.names[i]=fields.names[i+1];
	fields.data[i]=fields.data[i+1];
	}
}


void init_fields(void)
{
init_namespace(&fields);
fields.add_object=(int (*)(char *,void *))add_field;
}

void reset_fields(void)
{
long i;
for(i=fields.names_free-1;i>=0;i--)
	delete_field(i);	
}


AGENT * get_agent(FIELD *f, LOCATION l)
{
long x,y;

x=X(l)+X(f->current);
y=Y(l)+Y(f->current);
if(x<0)return NULL;
if(y<0)return NULL;
if(x>=f->x_size)return NULL;
if(y>=f->y_size)return NULL;
return f->agent[x][y];
}

FIELD * new_field(long x,long y)
{
long i,j;
FIELD *f;

f=do_alloc(1,sizeof(FIELD));
f->x_size=x;
f->y_size=y;
f->id=get_unique_number();
f->agent=do_alloc(x,sizeof(void *));
for(i=0;i<x;i++){
	f->agent[i]=do_alloc(y,sizeof(void *));
	for(j=0;j<y;j++)f->agent[i][j]=NULL;
	}
f->name=strdup("");
f->neighbor=NULL;
f->num_neighbors=0;
f->current=LOC(0,0);
f->shift_matrix=NULL;
f->shift_matrix_x_size=0;
f->shift_matrix_y_size=0;
f->seeding_commands=do_alloc(10,sizeof(SEEDING_COMMAND *));
f->seeding_commands_size=10;
f->seeding_commands_free=0;
f->seeded=0;
return f;
}

void free_field(FIELD *f)
{
long i;
clear_field(f);
free(f->name);
if(f->neighbor!=NULL)free(f->neighbor);
if(f->shift_matrix!=NULL){
	for(i=0;i<f->shift_matrix_x_size;i++)free(f->shift_matrix[i]);
	free(f->shift_matrix);	
	}
for(i=0;i<f->seeding_commands_free;i++)free(f->seeding_commands[i]);
free(f->seeding_commands);
}

void expand_seeding_commands(FIELD *f)
{
SEEDING_COMMAND **sc;
long i;
sc=do_alloc(f->seeding_commands_size*2,sizeof(SEEDING_COMMAND *));
for(i=0;i<f->seeding_commands_free;i++)
	sc[i]=f->seeding_commands[i];
free(f->seeding_commands);
f->seeding_commands=sc;
f->seeding_commands_size=f->seeding_commands_size*2;	
}

void add_rect_seeding_command(FIELD *f,char *name, long x1, long y1, long x2,long y2, long percent)
{
SEEDING_COMMAND *sc;
sc=do_alloc(1,sizeof(SEEDING_COMMAND));
sc->rect.name=strdup(name);
sc->rect.type=SEEDING_COMMAND_RECT;
sc->rect.percent=percent;
sc->rect.x1=x1;
sc->rect.y1=y1;
sc->rect.x2=x2;
sc->rect.y2=y2;
if(f->seeding_commands_free+1>=f->seeding_commands_size)
	expand_seeding_commands(f);
f->seeding_commands[f->seeding_commands_free]=sc;
f->seeding_commands_free++;	
}

void add_circle_seeding_command(FIELD *f,char *name, long x1, long y1, long r, long percent)
{
SEEDING_COMMAND *sc;
sc=do_alloc(1,sizeof(SEEDING_COMMAND));
sc->circle.name=strdup(name);
sc->circle.type=SEEDING_COMMAND_CIRCLE;
sc->circle.percent=percent;
sc->circle.x1=x1;
sc->circle.y1=y1;
sc->circle.r=r;
if(f->seeding_commands_free+1>=f->seeding_commands_size)
	expand_seeding_commands(f);
f->seeding_commands[f->seeding_commands_free]=sc;
f->seeding_commands_free++;	
}

FIELD *add_new_field(char *name,long x,long y)
{
FIELD *f;
f=new_field(x,y);
fields.add_object(name,f);
return f;
}

void dump_field(FILE *fout, FIELD *f)
{
long i,k;
SEEDING_COMMAND *sc;
fprintf(fout,"field \"%s\"\n",f->name);
if(f!=NULL){
	fprintf(fout,"\tdimensions %ld %ld\n",f->x_size,f->y_size);
	}
for(k=0;k<f->seeding_commands_free;k++){
	sc=f->seeding_commands[k];
	switch(sc->common.type){
		case SEEDING_COMMAND_RECT:
			fprintf(fout,"\tpaint \"%s\" rect(%ld,%ld,%ld,%ld) %ld\n",
				sc->rect.name,sc->rect.x1,sc->rect.y1,sc->rect.x2,sc->rect.y2,sc->rect.percent);
			break;
		case SEEDING_COMMAND_CIRCLE:
			fprintf(fout,"\tpaint \"%s\" circle(%ld,%ld,%ld) %ld\n",
				sc->circle.name,sc->circle.x1,sc->circle.y1,sc->circle.r,sc->circle.percent);
			break;
		default:
			fprintf(fout,"\tpaint \"%s\"  unknown %ld\n",sc->common.name,sc->common.percent);
		}
	}
	
if(f->seeded){
	fprintf(fout,"\tdata\n");
	for(i=0;i<f->x_size;i++)
		for(k=0;k<f->y_size;k++)
			{
			fprintf(fout,"\t%ld %ld ",i,k);
			dump_agent(fout,f->agent[i][k]);
			fprintf(fout,"\n");
			}
	fprintf(fout,"\tend\n");
	}
fprintf(fout,"end\n");
}

void dump_fields(FILE *fout)
{
long i;
for(i=0;i<fields.names_free;i++){
	dump_field(fout,(FIELD *)fields.data[i]);
	}	
}

void paint_rectangle(char *agentclassname,FIELD *f, long x1,long y1,long x2,long y2, int percent)
{
AGENTCLASS *ac;
long i,j;
ac=get_agentclass(agentclassname);
if(ac==NULL)return;
if(x1<0)x1=0;
if(y1<0)y1=0;
if(x2+1>f->x_size)x2=f->x_size-1;
if(y2+1>f->y_size)y2=f->y_size-1;
for(i=x1;i<=x2;i++)
	for(j=y1;j<=y2;j++){
		if(make_random_int(1,100)<=percent){
			if(f->agent[i][j]!=NULL)free_agent(f->agent[i][j]);
			f->agent[i][j]=make_agent(ac);
			}
		}

}

void paint_circle(char *agentclassname,FIELD *f, long x1,long y1,long r, int percent)
{
AGENTCLASS *ac;
long i,j;
long a1,a2,b1,b2;
ac=get_agentclass(agentclassname);
if(ac==NULL)return;
a1=x1-r;
a2=x1+r;
b1=y1-r;
b2=y1+r;
if(a1<0)a1=0;
if(b1<0)b1=0;
if(a2+1>f->x_size)a2=f->x_size-1;
if(b2+1>f->y_size)b2=f->y_size-1;
for(i=a1;i<=a2;i++)
	for(j=b1;j<=b2;j++){
		if(SQR(i-x1)+SQR(j-y1)<=SQR(r))
		if(make_random_int(1,100)<=percent){
			if(f->agent[i][j]!=NULL)free_agent(f->agent[i][j]);
			f->agent[i][j]=make_agent(ac);
			}
		}

}

void clear_field(FIELD *f)
{
long i,j;
for(i=0;i<f->x_size;i++)
	for(j=0;j<f->y_size;j++){
		if(f->agent[i][j]!=NULL)free_agent(f->agent[i][j]);
		f->agent[i][j]=NULL;
		}
}

void execute_seeding_commands(FIELD *f)
{
long i;
SEEDING_COMMAND *sc;
for(i=0;i<f->seeding_commands_free;i++){
	sc=f->seeding_commands[i];
	switch(sc->common.type){
		case SEEDING_COMMAND_RECT:
			paint_rectangle(sc->rect.name,f,sc->rect.x1,sc->rect.y1,sc->rect.x2,sc->rect.y2,sc->rect.percent);
			break;
		case SEEDING_COMMAND_CIRCLE:
			paint_circle(sc->circle.name,f,sc->circle.x1,sc->circle.y1,sc->circle.r,sc->circle.percent);
			break;
		default:
			fprintf(stderr,"Uknown seeding command (%ld,\"%s\",%ld) in \"%s\"\n",
				sc->common.type,sc->common.name,sc->common.percent,
				f->name);
			break;
		}

	}
f->seeded=1;
}

void seed_fields(void)
{
long i;
for(i=0;i<fields.names_free;i++)
	if(fields.data[i]!=NULL){
		clear_field(fields.data[i]);
		execute_seeding_commands(fields.data[i]);
		}
}


long num_fields(void)
{
return fields.names_free;
}

FIELD *get_field(long index)
{
return (FIELD *)fields.data[index];
}
